# Copyright 2023 Michael Tietz (MT Software) <mtietz@mt-software.de>
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html)
from datetime import datetime, timedelta

from odoo import api, models


class SaleOrder(models.Model):
    _inherit = "sale.order"

    @api.model
    def cron_generate_daily_invoices(self):
        """Cron called daily to check if daily invoicing needs to be done."""
        company_ids = self._company_daily_invoicing()
        if company_ids:
            self.generate_daily_invoices(company_ids)

    @api.model
    def generate_daily_invoices(self, company_ids):
        """Generate daily invoices for customers who require that mode.

        Invoices will be generated by other jobs split for different customer
        and different payment term.
        """
        return self.generate_invoices_by_invoice_mode(
            company_ids,
            "daily",
            self._get_groupby_fields_for_daily_invoicing(),
            "invoicing_mode_daily_last_execution",
        )

    @api.model
    def _company_daily_invoicing(self):
        """Get company ids for which today is daily invoicing day."""
        last_max_date = datetime.now() - timedelta(days=1)
        domain = [
            "|",
            ("invoicing_mode_daily_last_execution", "<=", last_max_date),
            ("invoicing_mode_daily_last_execution", "=", False),
        ]
        return self.env["res.company"].search(domain)

    @api.model
    def _get_groupby_fields_for_daily_invoicing(self):
        """Returns the sale order fields used to group them into jobs."""
        return ["partner_invoice_id", "payment_term_id"]
