# Copyright 2020 Camptocamp SA
# Copyright 2023 Michael Tietz (MT Software) <mtietz@mt-software.de>
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html)
from datetime import datetime

from odoo import api, fields, models


class SaleOrder(models.Model):
    _inherit = "sale.order"

    invoicing_mode = fields.Selection(related="partner_invoice_id.invoicing_mode")

    @api.model
    def _generate_invoices_by_partner(self, saleorder_ids):
        """Generate invoices for a group of sale order belonging to a customer."""
        sales = (
            self.browse(saleorder_ids)
            .exists()
            .filtered(lambda r: r.invoice_status == "to invoice")
        )
        if not sales:
            return "No sale order found to invoice ?"
        sales.partner_invoice_id.ensure_one()
        invoices = sales._create_invoices(
            grouped=sales.partner_invoice_id.one_invoice_per_order,
            final=True,
        )
        for invoice in invoices:
            invoice.with_delay()._validate_invoice()
        return invoices

    @api.model
    def generate_invoices_by_invoice_mode(
        self,
        companies=None,
        invoice_mode=None,
        groupby=None,
        last_execution_field_name=None,
    ):
        """Generate weekly invoices for customers who require that mode.

        Invoices will be generated by other jobs split for different customer
        and different payment term.
        """
        if not invoice_mode:
            return self.env[self._name]
        if not companies:
            companies = self.company_id
        saleorder_groups = self.read_group(
            [
                ("invoicing_mode", "=", invoice_mode),
                ("invoice_status", "=", "to invoice"),
                ("company_id", "in", companies.ids),
            ],
            ["partner_invoice_id"],
            groupby=groupby,
            lazy=False,
        )
        for saleorder_group in saleorder_groups:
            saleorder_ids = self.search(saleorder_group["__domain"]).ids
            self.with_delay()._generate_invoices_by_partner(saleorder_ids)
        companies.write({last_execution_field_name: datetime.now()})
        return saleorder_groups

    def _create_invoices(self, grouped=False, final=False, date=None):
        moves = self.env["account.move"]
        for partner_invoice in self.partner_invoice_id:
            sales = self.filtered(lambda r: r.partner_invoice_id == partner_invoice)
            moves += super(SaleOrder, sales)._create_invoices(
                grouped=partner_invoice.one_invoice_per_order,
                final=final,
                date=date,
            )
        return moves
