# Copyright 2018 Therp BV <https://therp.nl>
# Copyright 2022 Hunki Enterprises BV <https://hunki-enterprises.com>
# License AGPL-3.0 or later (https://www.gnu.org/licenses/agpl.html).
from odoo.tests.common import TransactionCase
from odoo.tools.mail import generate_tracking_message_id


class TestMailAutogeneratedHeader(TransactionCase):
    def setUp(self):
        super().setUp()
        self.mail = self.env["mail.mail"].create(
            {
                "subject": "testmessage",
                "email_from": "test@test.com",
                "email_to": "test@test.com",
                "message_id": "message_id",
            }
        )
        self.message = self.env["ir.mail_server"].build_email(
            [self.mail.email_from],
            [self.mail.email_to],
            self.mail.subject,
            "",
            message_id=self.mail.message_id,
        )

    def test_sending(self):
        """Test that sending a mail has the Auto-Submitted header"""
        self.env["ir.mail_server"].send_email(self.message)
        self.assertEqual(self.message["Auto-Submitted"], "auto-generated")

    def test_receiving(self):
        """Test that receiving mails with some auto submitted marker won't
        cause new notifications being sent"""
        demo_user = self.env.ref("base.user_demo")
        self.message.replace_header(
            "Message-Id",
            generate_tracking_message_id(42),
        )

        self.env["mail.notification"].search(
            [("res_partner_id", "=", demo_user.partner_id.id)]
        ).unlink()

        partner_id = self.env["mail.thread"].message_process(
            "res.partner",
            self.message.as_string(),
        )
        partner = self.env["res.partner"].browse(partner_id)
        partner.message_subscribe(partner_ids=demo_user.partner_id.ids)
        reply = self.message
        reply["References"] = self.message["Message-Id"]
        reply.replace_header("Message-Id", "message_id3")
        thread_id = self.env["mail.thread"].message_process(
            "res.partner",
            reply.as_string(),
        )
        self.assertEqual(thread_id, partner.id)
        notifications = self.env["mail.notification"].search(
            [("res_partner_id", "=", demo_user.partner_id.id)]
        )
        # mail is not autogenerated, should have generated mails
        self.assertTrue(notifications.notification_type == "email")
        notifications.unlink()

        reply.replace_header("Message-Id", "message_id4")
        reply["Auto-Submitted"] = "auto-generated"
        thread_id = self.env["mail.thread"].message_process(
            "res.partner",
            reply.as_string(),
        )
        self.assertEqual(thread_id, partner.id)
        notifications = self.env["mail.notification"].search(
            [("res_partner_id", "=", demo_user.partner_id.id)]
        )
        # mail is autogenerated, shouldn't have generated mails
        self.assertFalse(notifications.notification_type == "email")
