# Copyright (C) 2019 Brian McMaster
# Copyright (C) 2019 Open Source Integrators
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl).
from odoo import api, fields, models


class SaleOrderLine(models.Model):
    _inherit = "sale.order.line"

    qty_delivered_method = fields.Selection(
        selection_add=[("field_service", "Field Service Order")],
        ondelete={"field_service": "cascade"},
    )
    fsm_order_id = fields.Many2one(
        "fsm.order",
        "Order",
        index=True,
        copy=False,
        help="Field Service Order generated by the sales order item",
    )

    @api.depends("product_id.type")
    def _compute_product_updatable(self):
        for line in self:
            if line.product_id.type == "service" and line.state == "sale":
                line.product_updatable = False
            else:
                super(SaleOrderLine, line)._compute_product_updatable()

    @api.depends("product_id")
    def _compute_qty_delivered_method(self):
        super(SaleOrderLine, self)._compute_qty_delivered_method()
        for line in self:
            if not line.is_expense and line.product_id.field_service_tracking == "line":
                line.qty_delivered_method = "field_service"

    @api.depends("fsm_order_id.stage_id")
    def _compute_qty_delivered(self):
        super(SaleOrderLine, self)._compute_qty_delivered()
        lines_by_fsm = self.filtered(
            lambda sol: sol.qty_delivered_method == "field_service"
        )
        complete = self.env.ref("fieldservice.fsm_stage_completed")
        for line in lines_by_fsm:
            qty = 0
            if line.fsm_order_id.stage_id == complete:
                qty = line.product_uom_qty
                line.qty_delivered = qty

    @api.model
    def create(self, values):
        line = super(SaleOrderLine, self).create(values)
        if line.state == "sale":
            line.order_id._field_service_generation()
        return line

    def _get_invoiceable_fsm_order_domain(self):
        """
        Override this method to define more search criteria for invoiceable
        fsm order
        :return:
        """
        invoiceable_stage_ids = self.env["fsm.stage"]._get_invoiceable_stage()
        dom = [
            "|",
            ("sale_line_id", "=", self.id),
            ("sale_id", "=", self.order_id.id),
            ("invoice_lines", "=", False),
        ]
        if invoiceable_stage_ids:
            dom.append(("stage_id", "in", invoiceable_stage_ids.ids))
        return dom

    def _get_invoiceable_fsm_order(self):
        dom = self._get_invoiceable_fsm_order_domain()
        return self.env["fsm.order"].search(dom)

    def _prepare_invoice_line(self, **optional_values):
        res = super()._prepare_invoice_line(**optional_values)
        if self.fsm_order_id:
            fsm_orders = self._get_invoiceable_fsm_order()
            if fsm_orders:
                res.update({"fsm_order_ids": [(6, 0, fsm_orders.ids)]})
        return res
