# Copyright (C) 2019 - TODAY, Brian McMaster, Open Source Integrators
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl).

from odoo import api, fields, models, _


class SaleOrder(models.Model):
    _inherit = 'sale.order'

    fsm_location_id = fields.Many2one(
        'fsm.location', string='Service Location',
        help="SO Lines generating a FSM order will be for this location")
    date_fsm_request = fields.Datetime(
        string='Requested Service Date', readonly=True,
        states={'draft': [('readonly', False)], 'sent': [('readonly', False)]},
        copy=False, default=fields.Datetime.now)
    fsm_order_ids = fields.Many2many(
        'fsm.order', compute='_compute_fsm_order_ids',
        string='Field Service orders associated to this sale')
    fsm_order_count = fields.Float(
        string='Field Service Orders', compute='_compute_fsm_order_ids')
    fsm_recurring_ids = fields.Many2many(
        'fsm.recurring', compute='_compute_fsm_recurring_ids',
        string='Field Service Recurring orders associated to this sale')
    fsm_recurring_count = fields.Float(
        string='FSM Recurring Orders', compute='_compute_fsm_recurring_ids')

    @api.multi
    @api.depends('order_line.product_id')
    def _compute_fsm_order_ids(self):
        for order in self:
            order.fsm_order_ids = self.env['fsm.order'].search([
                ('sale_line_id', 'in', order.order_line.ids)])
            order.fsm_order_count = len(order.fsm_order_ids)

    @api.multi
    @api.depends('order_line.product_id')
    def _compute_fsm_recurring_ids(self):
        for order in self:
            order.fsm_recurring_ids = self.env['fsm.recurring'].search([
                ('sale_line_id', 'in', order.order_line.ids)])
            order.fsm_recurring_count = len(order.fsm_recurring_ids)

    @api.multi
    def action_confirm(self):
        """ On SO confirmation, some lines generate field service orders. """
        result = super(SaleOrder, self).action_confirm()
        self.order_line._field_service_generation()
        return result

    @api.multi
    def action_view_fsm_order(self):
        fsm_orders = self.mapped('fsm_order_ids')
        action = self.env.ref('fieldservice.action_fsm_dash_order').read()[0]
        if len(fsm_orders) > 1:
            action['domain'] = [('id', 'in', fsm_orders.ids)]
        elif len(fsm_orders) == 1:
            action['views'] = [(self.env.
                                ref('fieldservice.fsm_order_form').id,
                                'form')]
            action['res_id'] = fsm_orders.id
        else:
            action = {'type': 'ir.actions.act_window_close'}
        return action

    @api.multi
    def action_view_fsm_recurring(self):
        fsm_recurrings = self.mapped('fsm_recurring_ids')
        action = self.env.ref(
            'fieldservice_recurring.action_fsm_recurring').read()[0]
        if len(fsm_recurrings) > 1:
            action['domain'] = [('id', 'in', fsm_recurrings.ids)]
        elif len(fsm_recurrings) == 1:
            action['views'] = [
                (self.env.ref(
                    'fieldservice_recurring.fsm_recurring_form_view').id,
                    'form')]
            action['res_id'] = fsm_recurrings.id
        else:
            action = {'type': 'ir.actions.act_window_close'}
        return action


class SaleOrderLine(models.Model):
    _inherit = "sale.order.line"

    fsm_order_id = fields.Many2one(
        'fsm.order', 'Order', index=True,
        help="Field Service Order generated by the sales order item")
    fsm_recurring_id = fields.Many2one(
        'fsm.recurring', 'Recurring Order', index=True,
        help="Field Service Recurring Order generated by the sale order line")
    is_field_service = fields.Boolean(
        "Is a Field Service", compute='_compute_is_field_service',
        store=True,
        help="""Sales Order item should generate an Order and/or
                Order Recurrence, depending on the product settings.""")

    @api.multi
    @api.depends('product_id.type')
    def _compute_is_field_service(self):
        for so_line in self:
            so_line.is_field_service = so_line.product_id.type == 'service'

    @api.depends('product_id.type')
    def _compute_product_updatable(self):
        for line in self:
            if line.product_id.type == 'service' and line.state == 'sale':
                line.product_updatable = False
            else:
                super(SaleOrderLine, line)._compute_product_updatable()

    @api.model
    def create(self, values):
        line = super(SaleOrderLine, self).create(values)
        if line.state == 'sale':
            line._field_service_generation()
        return line

    def _field_create_fsm_order_prepare_values(self):
        self.ensure_one()
        return {
            'customer_id': self.order_id.partner_id.id,
            'location_id': self.order_id.fsm_location_id.id,
            # 'request_early': ,
            'scheduled_date_start': self.order_id.date_fsm_request,
            'description': self.name,
            'template_id': self.product_id.fsm_order_template_id.id,
            'sale_line_id': self.id,
            'company_id': self.company_id.id,
        }

    def _field_create_fsm_recurring_prepare_values(self):
        self.ensure_one()
        template = self.product_id.fsm_recurring_template_id
        note = self.name
        if template.description:
            note += '\n ' + template.description
        return {
            'customer_id': self.order_id.partner_id.id,
            'location_id': self.order_id.fsm_location_id.id,
            'start_date': self.order_id.date_fsm_request,
            'fsm_recurring_template_id': template.id,
            'description': note,
            'max_orders': template.max_orders,
            'fsm_frequency_set_id': template.fsm_frequency_set_id.id,
            'fsm_order_template_id': template.fsm_order_template_id.id,
            'sale_line_id': self.id,
            'company_id': self.company_id.id,
        }

    @api.multi
    def _field_create_fsm_order(self):
        """ Generate fsm_order for the given so line, and link it.
            :return a mapping with the so line id and its linked fsm_order
            :rtype dict
        """
        result = {}
        for so_line in self:
            # create fsm_order
            values = so_line._field_create_fsm_order_prepare_values()
            fsm_order = self.env['fsm.order'].sudo().create(values)
            so_line.write({'fsm_order_id': fsm_order.id})
            # post message on SO
            msg_body = _(
                """Field Service Order Created (%s): <a href=
                   # data-oe-model=fsm.order data-oe-id=%d>%s</a>
                """) % (so_line.product_id.name, fsm_order.id, fsm_order.name)
            so_line.order_id.message_post(body=msg_body)
            # post message on fsm_order
            fsm_order_msg = _(
                """This order has been created from: <a href=
                   # data-oe-model=sale.order data-oe-id=%d>%s</a> (%s)
                """) % (so_line.order_id.id, so_line.order_id.name,
                        so_line.product_id.name)
            fsm_order.message_post(body=fsm_order_msg)
            result[so_line.id] = fsm_order
        return result

    @api.multi
    def _field_create_fsm_recurring(self):
        """ Generate fsm_recurring for the given so line, and link it.
            :return a mapping with the so line id and its linked fsm_recurring
            :rtype dict
        """
        result = {}
        for so_line in self:
            # create fsm_recurring
            values = so_line._field_create_fsm_recurring_prepare_values()
            fsm_recurring = self.env['fsm.recurring'].sudo().create(values)
            fsm_recurring.action_start()
            so_line.write({'fsm_recurring_id': fsm_recurring.id})
            # post message on SO
            msg_body = _(
                """Field Service recurring Created (%s): <a href=
                   # data-oe-model=fsm.recurring data-oe-id=%d>%s</a>
                """) % (so_line.product_id.name,
                        fsm_recurring.id,
                        fsm_recurring.name)
            so_line.order_id.message_post(body=msg_body)
            # post message on fsm_recurring
            fsm_recurring_msg = _(
                """This recurring has been created from: <a href=
                   # data-oe-model=sale.order data-oe-id=%d>%s</a> (%s)
                """) % (so_line.order_id.id, so_line.order_id.name,
                        so_line.product_id.name)
            fsm_recurring.message_post(body=fsm_recurring_msg)
            result[so_line.id] = fsm_recurring
        return result

    @api.multi
    def _field_find_fsm_order(self):
        """ Find the fsm_order generated by the so lines. If no fsm_order
            linked, it will be created automatically.
            :return a mapping with the so line id and its linked fsm_order
            :rtype dict
        """
        # one search for all so lines
        fsm_orders = self.env['fsm.order'].search([
            ('sale_line_id', 'in', self.ids)])
        fsm_order_sol_mapping = {
            fsm_order.sale_line_id.id: fsm_order for fsm_order in fsm_orders}
        result = {}
        for so_line in self:
            # If the SO was confirmed, cancelled, set to draft then confirmed,
            # avoid creating a new fsm_order.
            fsm_order = fsm_order_sol_mapping.get(so_line.id)
            # If not found, create one fsm_order for the so line
            if not fsm_order:
                fsm_order = so_line._field_create_fsm_order()[so_line.id]
            result[so_line.id] = fsm_order
        return result

    @api.multi
    def _field_find_fsm_recurring(self):
        """ Find the fsm_recurring generated by the so lines. If no
            fsm_recurring linked, it will be created automatically.
            :return a mapping with the so line id and its linked
            fsm_recurring
            :rtype dict
        """
        # one search for all so lines
        fsm_recurrings = self.env['fsm.recurring'].search([
            ('sale_line_id', 'in', self.ids)])
        fsm_recurring_sol_mapping = {
            fsm_recurring.sale_line_id.id:
                fsm_recurring for fsm_recurring in fsm_recurrings}
        result = {}
        for so_line in self:
            # If the SO was confirmed, cancelled, set to draft then confirmed,
            # avoid creating a new fsm_recurring.
            fsm_recurring = fsm_recurring_sol_mapping.get(so_line.id)
            # If not found, create one fsm_recurring for the so line
            if not fsm_recurring:
                fsm_recurring = so_line._field_create_fsm_recurring(
                    )[so_line.id]
            result[so_line.id] = fsm_recurring
        return result

    @api.multi
    def _field_service_generation(self):
        """ For service lines, create the field service order. If it already
            exists, it simply links the existing one to the line.
        """
        for so_line in self.filtered(lambda sol: sol.is_field_service):
            # create order
            if so_line.product_id.field_service_tracking == 'order':
                so_line._field_find_fsm_order()
            # create recurring order
            elif so_line.product_id.field_service_tracking == 'recurring':
                so_line._field_find_fsm_recurring()
