# See LICENSE file for full copyright and licensing details.

from odoo import api, fields, models
from odoo.osv import expression


class HotelRoomType(models.Model):
    _name = "hotel.room.type"
    _description = "Room Type"

    categ_id = fields.Many2one("hotel.room.type", "Category")
    child_ids = fields.One2many(
        "hotel.room.type", "categ_id", "Child Categories"
    )
    product_categ_id = fields.Many2one(
        "product.category", "Product Category", delegate=True, required=True
    )

    @api.model
    def create(self, vals):
        if vals.get("categ_id", False):
            room_categ = self.env["hotel.room.type"].browse(
                vals.get("categ_id")
            )
            vals.update({"parent_id": room_categ.product_categ_id.id})
        return super(HotelRoomType, self).create(vals)

    @api.multi
    def write(self, vals):
        if vals.get("categ_id", False):
            room_categ = self.env["hotel.room.type"].browse(
                vals.get("categ_id")
            )
            vals.update({"parent_id": room_categ.product_categ_id.id})
        return super(HotelRoomType, self).write(vals)

    @api.multi
    def name_get(self):
        def get_names(cat):
            """ Return the list [cat.name, cat.categ_id.name, ...] """
            res = []
            while cat:
                res.append(cat.name)
                cat = cat.categ_id
            return res

        return [(cat.id, " / ".join(reversed(get_names(cat)))) for cat in self]

    @api.model
    def name_search(self, name, args=None, operator="ilike", limit=100):
        if not args:
            args = []
        if name:
            # Be sure name_search is symmetric to name_get
            category_names = name.split(" / ")
            parents = list(category_names)
            child = parents.pop()
            domain = [("name", operator, child)]
            if parents:
                names_ids = self.name_search(
                    " / ".join(parents),
                    args=args,
                    operator="ilike",
                    limit=limit,
                )
                category_ids = [name_id[0] for name_id in names_ids]
                if operator in expression.NEGATIVE_TERM_OPERATORS:
                    categories = self.search([("id", "not in", category_ids)])
                    domain = expression.OR(
                        [[("categ_id", "in", categories.ids)], domain]
                    )
                else:
                    domain = expression.AND(
                        [[("categ_id", "in", category_ids)], domain]
                    )
                for i in range(1, len(category_names)):
                    domain = [
                        [
                            (
                                "name",
                                operator,
                                " / ".join(category_names[-1 - i :]),
                            )
                        ],
                        domain,
                    ]
                    if operator in expression.NEGATIVE_TERM_OPERATORS:
                        domain = expression.AND(domain)
                    else:
                        domain = expression.OR(domain)
            categories = self.search(
                expression.AND([domain, args]), limit=limit
            )
        else:
            categories = self.search(args, limit=limit)
        return categories.name_get()
