# Copyright 2013-Today Odoo SA
# Copyright 2016 Chafique DELLI @ Akretion
# Copyright 2018 Tecnativa - Carlos Dauden
# License AGPL-3.0 or later (https://www.gnu.org/licenses/agpl).

from odoo.tests import common
from odoo.exceptions import AccessError, UserError
from odoo.modules.module import get_resource_path
from odoo.tools import convert_file


class TestPurchaseSaleInterCompany(common.SavepointCase):

    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls._load(
            "account_invoice_inter_company",
            "tests",
            "inter_company_invoice.xml",
        )
        cls._load(
            "purchase_sale_inter_company",
            "tests",
            "inter_company_purchase_sale.xml",
        )
        cls.warehouse_company_a = cls.env.ref(
            'purchase_sale_inter_company.warehouse_company_a')
        cls.purchase_company_a = cls.env.ref(
            'purchase_sale_inter_company.purchase_company_a')
        # Test demo data is create by admin, thus the purchase order default
        # picking type has to be fixed.
        cls.purchase_company_a.picking_type_id = (
            cls.warehouse_company_a.in_type_id)
        cls.company_a = cls.env.ref('purchase_sale_inter_company.company_a')
        cls.company_b = cls.env.ref('purchase_sale_inter_company.company_b')
        cls.company_b.so_from_po = True
        cls.user_a = cls.env.ref('purchase_sale_inter_company.user_company_a')
        cls.user_b = cls.env.ref('purchase_sale_inter_company.user_company_b')
        cls.intercompany_user = cls.user_b.copy()
        cls.intercompany_user.company_ids |= cls.company_a
        cls.company_b.intercompany_user_id = cls.intercompany_user
        cls.account_sale_b = cls.env.ref(
            'purchase_sale_inter_company.a_sale_company_b')
        cls.product_consultant = cls.env.ref(
            'purchase_sale_inter_company.product_consultant_multi_company')
        cls.product_consultant.sudo(
            cls.user_b.id).property_account_income_id = cls.account_sale_b

    @classmethod
    def _load(cls, module, *args):
        convert_file(
            cls.cr, "purchase_sale_inter_company",
            get_resource_path(module, *args),
            {}, 'init', False, 'test', cls.registry._assertion_report,
        )

    def test_purchase_sale_inter_company(self):
        self.purchase_company_a.notes = 'Test note'
        # Confirm the purchase of company A
        self.purchase_company_a.sudo(self.user_a).button_approve()
        # Check sale order created in company B
        sales = self.env['sale.order'].sudo(self.user_b).search([
            ('auto_purchase_order_id', '=', self.purchase_company_a.id)
        ])
        self.assertNotEquals(sales, False)
        self.assertEquals(len(sales), 1)
        if sales.company_id.sale_auto_validation:
            self.assertEquals(sales.state, 'sale')
        else:
            self.assertEquals(sales.state, 'draft')
        self.assertEquals(sales.partner_id,
                          self.purchase_company_a.company_id.partner_id)
        self.assertEquals(sales.company_id.partner_id,
                          self.purchase_company_a.partner_id)
        self.assertEquals(len(sales.order_line),
                          len(self.purchase_company_a.order_line))
        self.assertEquals(sales.order_line.product_id,
                          self.purchase_company_a.order_line.product_id)
        self.assertEquals(sales.note, 'Test note')

    def xxtest_date_planned(self):
        # Install sale_order_dates module
        module = self.env['ir.module.module'].search(
            [('name', '=', 'sale_order_dates')])
        if not module:
            return False
        module.button_install()
        self.purchase_company_a.date_planned = '2070-12-31'
        self.purchase_company_a.sudo(self.user_a).button_approve()
        sales = self.env['sale.order'].sudo(self.user_b).search([
            ('auto_purchase_order_id', '=', self.purchase_company_a.id),
        ])
        self.assertEquals(sales.requested_date, '2070-12-31')

    def test_raise_product_access(self):
        product_rule = self.env.ref('product.product_comp_rule')
        product_rule.active = True
        self.product_consultant.company_id = self.company_b
        with self.assertRaises(AccessError):
            self.purchase_company_a.sudo(self.user_a).button_approve()

    def test_raise_currency(self):
        currency = self.purchase_company_a.currency_id
        currency = self.env['res.currency'].search(
            [('id', '!=', currency.id)], limit=1)
        self.purchase_company_a.currency_id = currency
        with self.assertRaises(UserError):
            self.purchase_company_a.sudo(self.user_a).button_approve()

    def test_purchase_invoice_relation(self):
        self.purchase_company_a.sudo(self.user_a).button_approve()
        sales = self.env['sale.order'].sudo(self.user_b).search([
            ('auto_purchase_order_id', '=', self.purchase_company_a.id),
        ])
        sale_invoice_id = sales.action_invoice_create()[0]
        sale_invoice = self.env['account.invoice'].browse(sale_invoice_id)
        sale_invoice.action_invoice_open()
        self.assertEquals(sale_invoice.auto_invoice_id,
                          self.purchase_company_a.invoice_ids)
        self.assertEquals(sale_invoice.auto_invoice_id.invoice_line_ids,
                          self.purchase_company_a.order_line.invoice_lines)

    def test_cancel(self):
        self.company_b.sale_auto_validation = False
        self.purchase_company_a.sudo(self.user_a).button_approve()
        sales = self.env['sale.order'].sudo(self.user_b).search([
            ('auto_purchase_order_id', '=', self.purchase_company_a.id)
        ])
        self.assertEquals(self.purchase_company_a.partner_ref, sales.name)
        self.purchase_company_a.sudo(self.user_a).button_cancel()
        self.assertFalse(self.purchase_company_a.partner_ref)

    def test_cancel_confirmed_po_so(self):
        self.company_b.sale_auto_validation = True
        self.purchase_company_a.sudo(self.user_a).button_approve()
        self.env['sale.order'].sudo(self.user_b).search([
            ('auto_purchase_order_id', '=', self.purchase_company_a.id)
        ])
        with self.assertRaises(UserError):
            self.purchase_company_a.sudo(self.user_a).button_cancel()
