#  Copyright 2024 Simone Rubino - Aion Tech
#  License AGPL-3.0 or later (https://www.gnu.org/licenses/agpl).

import base64
import tempfile

from odoo.modules import get_module_resource
from odoo.tests import Form
from odoo.tests.common import SingleTransactionCase


class FatturapaCommon(SingleTransactionCase):
    def getFile(self, filename, module_name=None):
        if module_name is None:
            module_name = "l10n_it_fatturapa_in"
        path = get_module_resource(module_name, "tests", "data", filename)
        with open(path, "rb") as test_data:
            with tempfile.TemporaryFile() as out:
                base64.encode(test_data, out)
                out.seek(0)
                return path, out.read()

    def create_wt(self):
        return self.env["withholding.tax"].create(
            {
                "name": "1040",
                "code": "1040",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 20.0})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.a").id,
            }
        )

    def create_wt_23_20(self):
        return self.env["withholding.tax"].create(
            {
                "name": "2320",
                "code": "2320",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 23.0, "base": 0.2})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.a").id,
            }
        )

    def create_wt_23_50(self):
        return self.env["withholding.tax"].create(
            {
                "name": "2320",
                "code": "2320",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 23.0, "base": 0.5})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.a").id,
            }
        )

    def create_wt_26_20q(self):
        return self.env["withholding.tax"].create(
            {
                "name": "2620q",
                "code": "2620q",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 26.0, "base": 0.2})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.q").id,
            }
        )

    def create_wt_26_40q(self):
        return self.env["withholding.tax"].create(
            {
                "name": "2640q",
                "code": "2640q",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 26.0, "base": 0.4})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.q").id,
            }
        )

    def create_wt_27_20q(self):
        return self.env["withholding.tax"].create(
            {
                "name": "2720q",
                "code": "2720q",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 27.0, "base": 0.2})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.q").id,
            }
        )

    def create_wt_4q(self):
        return self.env["withholding.tax"].create(
            {
                "name": "4q",
                "code": "4q",
                "wt_types": "enasarco",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account_id,
                "payment_term": self.env.ref(
                    "account.account_payment_term_immediate"
                ).id,
                "rate_ids": [(0, 0, {"tax": 4.0, "base": 1.0})],
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.q").id,
            }
        )

    @classmethod
    def create_misc_journal(cls):
        return cls.env["account.journal"].create(
            {
                "name": "Test Miscellaneous Journal",
                "code": "TMJ",
                "type": "general",
            }
        )

    def create_wt_115_r(self):
        return self.env["withholding.tax"].create(
            {
                "name": "1040 R",
                "code": "1040R",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account.id,
                "journal_id": self.misc_journal.id,
                "payment_term": self.env.ref("account.account_payment_term_advance").id,
                "wt_types": "ritenuta",
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.r").id,
                "rate_ids": [
                    (
                        0,
                        0,
                        {
                            "tax": 11.50,
                            "base": 1.0,
                        },
                    )
                ],
            }
        )

    def create_wt_enasarco_115_a(self):
        return self.env["withholding.tax"].create(
            {
                "name": "1040/3",
                "code": "1040",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account.id,
                "journal_id": self.misc_journal.id,
                "payment_term": self.env.ref("account.account_payment_term_advance").id,
                "wt_types": "ritenuta",
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.a").id,
                "rate_ids": [
                    (
                        0,
                        0,
                        {
                            "tax": 11.50,
                            "base": 1.0,
                        },
                    )
                ],
            }
        )

    def create_wt_enasarco_85_r(self):
        return self.env["withholding.tax"].create(
            {
                "name": "Enasarco 8,50",
                "code": "TC07",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account.id,
                "journal_id": self.misc_journal.id,
                "payment_term": self.env.ref("account.account_payment_term_advance").id,
                "wt_types": "enasarco",
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.r").id,
                "rate_ids": [
                    (
                        0,
                        0,
                        {
                            "tax": 8.5,
                            "base": 1.0,
                        },
                    )
                ],
            }
        )

    def create_wt_enasarco_157_r(self):
        return self.env["withholding.tax"].create(
            {
                "name": "Enasarco",
                "code": "TC07",
                "account_receivable_id": self.receivable_account.id,
                "account_payable_id": self.payable_account.id,
                "journal_id": self.misc_journal.id,
                "payment_term": self.env.ref("account.account_payment_term_advance").id,
                "wt_types": "enasarco",
                "payment_reason_id": self.env.ref("l10n_it_payment_reason.r").id,
                "rate_ids": [
                    (
                        0,
                        0,
                        {
                            "tax": 1.57,
                            "base": 1.0,
                        },
                    )
                ],
            }
        )

    @classmethod
    def create_receivable_account(cls):
        return cls.env["account.account"].create(
            {
                "name": "Test WH tax",
                "code": "whtaxrec2",
                "user_type_id": cls.env.ref("account.data_account_type_receivable").id,
                "reconcile": True,
            }
        )

    @classmethod
    def create_payable_account(cls):
        return cls.env["account.account"].create(
            {
                "name": "Test WH tax",
                "code": "whtaxpay2",
                "user_type_id": cls.env.ref("account.data_account_type_payable").id,
                "reconcile": True,
            }
        )

    def create_res_bank(self):
        return self.env["res.bank"].create(
            {
                "name": "Banca generica",
                "bic": "BCITITMM",
            }
        )

    def create_attachment(self, name, file_name, module_name=None):
        if module_name is None:
            module_name = "l10n_it_fatturapa_in"

        attach_form = Form(self.env["fatturapa.attachment.in"])
        attach_form.name = name
        attach_form.datas = self.getFile(file_name, module_name=module_name)[1]
        attach = attach_form.save()
        return attach

    def run_wizard(
        self, name, file_name, mode="import", wiz_values=None, module_name=None
    ):
        if module_name is None:
            module_name = "l10n_it_fatturapa_in"
        attach = self.create_attachment(name, file_name, module_name=module_name)
        attach_id = attach.id
        if mode == "import":
            wizard_form = Form(
                self.wizard_model.with_context(
                    active_ids=[attach_id], active_model="fatturapa.attachment.in"
                )
            )
            wizard = wizard_form.save()
            return wizard.importFatturaPA()
        if mode == "link":
            wizard_form = Form(
                self.wizard_link_model.with_context(
                    active_ids=[attach_id], active_model="fatturapa.attachment.in"
                )
            )
            wizard = wizard_form.save()
            if wiz_values:
                wiz_values.wizard_id = wizard
            return wizard.link()
        if not mode:
            # return created fatturapa.attachment.in record in case no mode provided
            return attach

    def run_wizard_multi(self, file_name_list, module_name=None):
        if module_name is None:
            module_name = "l10n_it_fatturapa_in"

        attachments = self.attach_model.create(
            [
                {
                    "name": file_name,
                    "datas": self.getFile(file_name, module_name=module_name)[1],
                }
                for file_name in file_name_list
            ]
        )

        wizard = self.wizard_model.with_context(
            active_model=attachments._name,
            active_ids=attachments.ids,
        ).create({})

        return wizard.importFatturaPA()

    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls.payable_account = cls.create_payable_account()
        cls.payable_account_id = cls.payable_account.id
        cls.receivable_account = cls.create_receivable_account()
        cls.misc_journal = cls.create_misc_journal()

    def setUp(self):
        super(FatturapaCommon, self).setUp()
        self.wizard_model = self.env["wizard.import.fatturapa"]
        self.wizard_link_model = self.env["wizard.link.to.invoice"]
        self.data_model = self.env["ir.model.data"]
        self.attach_model = self.env["fatturapa.attachment.in"]
        self.invoice_model = self.env["account.move"]
        self.headphones = self.env.ref("product.product_product_7_product_template")
        self.imac = self.env.ref("product.product_product_8_product_template")
        self.service = self.env.ref("product.product_product_1")
        arrotondamenti_attivi_account_id = (
            self.env["account.account"]
            .search(
                [
                    (
                        "user_type_id",
                        "=",
                        self.env.ref("account.data_account_type_other_income").id,
                    )
                ],
                limit=1,
            )
            .id
        )
        arrotondamenti_passivi_account_id = (
            self.env["account.account"]
            .search(
                [
                    (
                        "user_type_id",
                        "=",
                        self.env.ref("account.data_account_type_direct_costs").id,
                    )
                ],
                limit=1,
            )
            .id
        )
        arrotondamenti_tax_id = self.env["account.tax"].search(
            [("type_tax_use", "=", "purchase"), ("amount", "=", 0.0)],
            order="sequence",
            limit=1,
        )
        self.env.company.arrotondamenti_attivi_account_id = (
            arrotondamenti_attivi_account_id
        )
        self.env.company.arrotondamenti_passivi_account_id = (
            arrotondamenti_passivi_account_id
        )
        self.env.company.arrotondamenti_tax_id = arrotondamenti_tax_id
        self.env["res.lang"]._activate_lang("it_IT")
