# -*- coding: utf-8 -*-
#
#
#    Author: Guewen Baconnier, Jacques-Etienne Baudoux
#    Copyright 2013-2014 Camptocamp SA
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as
#    published by the Free Software Foundation, either version 3 of the
#    License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#
from openerp import models, fields, api
from openerp import exceptions
from openerp.tools.translate import _


class PurchaseOrder(models.Model):
    _inherit = 'purchase.order'

    transport_mode_id = fields.Many2one('transport.mode', 'Transport Mode')
    transport_vehicle_id = fields.Many2one('transport.vehicle',
                                           'Transport Vehicle')
    logistic_order_count = fields.Integer(compute='_count_logistic_order')

    @api.multi
    def _get_relevant_logistic_orders(self):
        return self.order_line.mapped('lo_line_ids.order_id')

    @api.one
    @api.depends('order_line.lo_line_ids')
    def _count_logistic_order(self):
        sales = self._get_relevant_logistic_orders()
        self.logistic_order_count = len(sales.ids)

    @api.multi
    def action_open_logistic_orders(self):
        sales = self._get_relevant_logistic_orders()
        action_ref = 'logistic_order.action_orders'
        action_dict = self.env.ref(action_ref).read()[0]
        action_dict['domain'] = [('id', 'in', sales.ids)]
        return action_dict


class PurchaseOrderLine(models.Model):
    _inherit = 'purchase.order.line'

    # one2one relation with selected_bid_line_id
    lr_source_line_id = fields.Many2one(
        'logistic.requisition.source',
        'Logistics Requisition Source',
        readonly=True,
        ondelete='restrict')
    from_bid_line_id = fields.Many2one(
        'purchase.order.line',
        'Generated from bid',
        readonly=True)
    po_line_from_bid_ids = fields.One2many(
        'purchase.order.line',
        'from_bid_line_id',
        'Lines generated by the bid',
        readonly=True)
    # reversal link to logistic order lines
    lo_line_ids = fields.One2many(
        comodel_name='sale.order.line',
        inverse_name='sourced_by')

    @api.model
    def _prepare_lrs_update_from_po_line(self, vals, po_line):
        """ Take the vals dict from po line and return a vals dict for LRS

        :param dict vals: value of to be written in new po line
        :param browse_record po_line: purchase.order.line
        :returns dict : vals to be written on logistic.requisition.source

        """
        lrs_vals = {}
        if vals.get('product_qty'):
            lrs_vals['proposed_qty'] = vals.get('product_qty')
        if vals.get('product_id'):
            lrs_vals['proposed_product_id'] = vals.get('product_id')
        if vals.get('product_uom'):
            lrs_vals['proposed_uom_id'] = vals.get('product_uom')
        if vals.get('price_unit'):
            to_curr = po_line.lr_source_line_id.requisition_id.currency_id
            from_curr = po_line.order_id.pricelist_id.currency_id
            price = from_curr.compute(vals.get('price_unit'), to_curr,
                                      round=False)
            lrs_vals['unit_cost'] = price
        return lrs_vals

    @api.multi
    def write(self, vals):
        """ Here we implement something to allow the update of LRS when some
        information are changed in PO line. It should be possible to do it
        when:
        PO is still in draft
        LRL is not marked as sourced
        Once done, nobody should be able to change the PO line infos
        """
        if (vals.get('product_qty') or
                vals.get('product_id') or
                vals.get('product_uom') or
                vals.get('price_unit')):
            for line in self:
                source_line = line.lr_source_line_id
                if (source_line and
                        not source_line.sourcing_method == 'reuse_bid'):
                    if (source_line.requisition_line_id.state in
                            ('sourced', 'quoted')):
                        raise exceptions.Warning(
                            _("You cannot change the informations because "
                              "this PO line is already linked to a Logistics "
                              "Requisition Line %s marked as sourced or "
                              "quoted.") % (line.lr_source_line_id.name)
                        )
                    else:
                        lrs_vals = self._prepare_lrs_update_from_po_line(
                            vals, line)
                        line.lr_source_line_id.write(lrs_vals)
        return super(PurchaseOrderLine, self).write(vals)

    @api.multi
    def unlink(self):
        for line in self:
            source_line = line.lr_source_line_id
            if source_line:
                if (source_line.requisition_line_id in ('sourced', 'quoted')):
                    raise Warning(
                        _("You cannot delete this PO line because it is "
                          "already linked to a Logistics Requisition Line %s "
                          "marked as sourced or quoted.")
                        % (line.lr_source_line_id.name)
                    )
        return super(PurchaseOrderLine, self).unlink()

    def copy_data(self, cr, uid, res_id, default=None, context=None):
        """ When copying purchase order lines to reuse a bid
        Link them to logistic requisition lines
        """
        if context is None:
            context = {}
        res = super(PurchaseOrderLine, self).copy_data(
            cr, uid, res_id, default=default, context=context)
        if 'reuse_from_source' in context:
            source_id = context['reuse_from_source'].get(res_id)
            res['lr_source_line_id'] = source_id
        return res
