# -*- coding: utf-8 -*-
# Copyright 2016 LasLabs Inc.
# Copyright 2017 Tecnativa - Jairo Llopis
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html).

import mock
from odoo.tests import common
from odoo import exceptions


@common.at_install(False)
@common.post_install(True)
class TestReport(common.HttpCase):
    def setUp(self):
        super(TestReport, self).setUp()
        self.Model = self.env['report']
        self.server = self.env['printing.server'].create({})
        self.report_vals = {}
        self.report_imd = self.env["ir.model.data"].create({
            "name": "test",
            "module": "base_report_to_printer",
            "model": "ir.ui.view",
        })
        self.report_view = self.env["ir.ui.view"].create({
            "name": "Test",
            "type": "qweb",
            "xml_id": "base_report_to_printer.test",
            "model_data_id": self.report_imd.id,
            "arch": """<t t-name="base_report_to_printer.test">
                <div>Test</div>
            </t>""",
        })
        self.report_imd.res_id = self.report_view.id
        self.report = self.env["ir.actions.report.xml"].create({
            "name": "Test",
            "report_type": "qweb-pdf",
            "model": "res.partner",
            "report_name": "base_report_to_printer.test",
        })
        self.partners = self.env["res.partner"]
        for n in range(5):
            self.partners += self.env["res.partner"].create({
                "name": "Test %d" % n,
            })

    def new_record(self):
        return self.Model.create(self.report_vals)

    def new_printer(self):
        return self.env['printing.printer'].create({
            'name': 'Printer',
            'server_id': self.server.id,
            'system_name': 'Sys Name',
            'default': True,
            'status': 'unknown',
            'status_message': 'Msg',
            'model': 'res.users',
            'location': 'Location',
            'uri': 'URI',
        })

    def test_can_print_report_context_skip(self):
        """ It should return False based on context """
        rec_id = self.new_record().with_context(
            must_skip_send_to_printer=True
        )
        res = rec_id._can_print_report(
            {'action': 'server'}, True, True
        )
        self.assertFalse(res)

    def test_can_print_report_true(self):
        """ It should return True when server print allowed """
        res = self.new_record()._can_print_report(
            {'action': 'server'}, True, True
        )
        self.assertTrue(res)

    def test_can_print_report_false(self):
        """ It should return False when server print not allowed """
        res = self.new_record()._can_print_report(
            {'action': 'server'}, True, False
        )
        self.assertFalse(res)

    def test_get_pdf_not_printable(self):
        """ It should print the report, only if it is printable
        """
        with mock.patch('odoo.addons.base_report_to_printer.models.'
                        'printing_printer.PrintingPrinter.'
                        'print_document') as print_document:
            self.Model.get_pdf(
                self.partners.ids, self.report.report_name)
            print_document.assert_not_called()

    def test_get_pdf_printable(self):
        """ It should print the report, only if it is printable
        """
        with mock.patch('odoo.addons.base_report_to_printer.models.'
                        'printing_printer.PrintingPrinter.'
                        'print_document') as print_document:
            self.report.property_printing_action_id.action_type = 'server'
            self.report.printing_printer_id = self.new_printer()
            document = self.Model.get_pdf(
                self.partners.ids, self.report.report_name)
            print_document.assert_called_once_with(
                self.report.report_name, document, self.report.report_type)

    def test_print_document_not_printable(self):
        """ It should print the report, regardless of the defined behaviour """
        self.report.printing_printer_id = self.new_printer()
        with mock.patch('odoo.addons.base_report_to_printer.models.'
                        'printing_printer.PrintingPrinter.'
                        'print_document') as print_document:
            self.Model.print_document(
                self.partners.ids, self.report.report_name)
            print_document.assert_called_once()

    def test_print_document_printable(self):
        """ It should print the report, regardless of the defined behaviour """
        self.report.property_printing_action_id.action_type = 'server'
        self.report.printing_printer_id = self.new_printer()
        with mock.patch('odoo.addons.base_report_to_printer.models.'
                        'printing_printer.PrintingPrinter.'
                        'print_document') as print_document:
            self.Model.print_document(
                self.partners.ids, self.report.report_name)
            print_document.assert_called_once()

    def test_print_document_no_printer(self):
        """ It should raise an error """
        with self.assertRaises(exceptions.UserError):
            self.Model.print_document(
                self.partners.ids, self.report.report_name)
