# -*- coding: utf-8 -*-
import json
import requests
from odoo import models, api


class SmsApi(models.AbstractModel):
    _inherit = 'sms.api'

    def _get_params_value(self):
        """
        Get values for spryng configuration ir.config_parameter
        """
        ir_config_param = self.env['ir.config_parameter'].sudo().get_param
        token = ir_config_param('spryng_sms.spryng_sms_token')
        url = ir_config_param('spryng_sms.spryng_sms_server_url')
        route = ir_config_param('spryng_sms.spryng_sms_route')
        originator = ir_config_param('spryng_sms.spryng_sms_originator')
        return token, url, route, originator
    
    def _get_headers_value(self):
        """
        Prepare header for request
        """
        token, url, route, originator = self._get_params_value()
        return {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': f'Bearer {token}'
        }

    def _sms_post_request(self, message, headers):
        ""
        token, url, route, originator = self._get_params_value()
        payload = json.dumps({
            "body": message['content'],
            "originator": originator,
            "recipients": [
                message['number']
            ],
            "route": route,
            "reference": str(message['res_id'])
        })
        return requests.request("POST", url, headers=headers, data=payload)

    @api.model
    def _send_sms_batch(self, messages):
        """
        Override of default Odoo function to send SMS messages
        ODOO URL: https://github.com/odoo/odoo/blob/944c946a7a5158b705f10cc11025e27467fb7db0/addons/sms/models/sms_api.py#L38
        """
        sms_provider_type = self.env['ir.config_parameter'].sudo().get_param('spryng_sms.sms_provider_type')
        if sms_provider_type == 'spryng_sms':
            return_values = []
            headers = self._get_headers_value()
            for message in messages:
                response = self._sms_post_request(message, headers)
                return_value = {
                    'res_id': message.get('res_id'),
                    'state': 'success',
                    'credit': 0,
                    'spryng': True
                }
                if response.status_code != 200:
                    sms_id = self.env['sms.sms'].sudo().browse(
                        message['res_id'])
                    error_code = json.dumps(
                        json.loads(response.text), indent=4)
                    sms_id.write({
                        'spryng_status_code': response.status_code,
                        'spryng_status_error': error_code,
                    })
                    sms_id.with_context(from_sms_api=True).action_generate_activity(
                        error_code, sms_id.mail_message_id.model, sms_id.mail_message_id.res_id, sms_id)
                    return_value.update({
                        'state': 'server_error'
                    })
                return_values.append(return_value)
            return return_values
        return super(SmsApi, self)._send_sms_batch(messages)
