# -*- coding: utf-8 -*-
# © 2008-2016 Camptocamp
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl).

import logging

from datetime import datetime
from odoo import fields, _
from odoo.exceptions import UserError

_logger = logging.getLogger(__name__)


class AbstractClassError(Exception):
    def __str__(self):
        return 'Abstract Class'

    def __repr__(self):
        return 'Abstract Class'


class AbstractMethodError(Exception):
    def __str__(self):
        return 'Abstract Method'

    def __repr__(self):
        return 'Abstract Method'


class UnknowClassError(Exception):
    def __str__(self):
        return 'Unknown Class'

    def __repr__(self):
        return 'Unknown Class'


class UnsuportedCurrencyError(Exception):
    def __init__(self, value):
        self.curr = value

    def __str__(self):
        return 'Unsupported currency %s' % self.curr

    def __repr__(self):
        return 'Unsupported currency %s' % self.curr


class CurrencyGetterType(type):
    """ Meta class for currency getters.
        Automaticaly registers new curency getter on class definition
    """
    getters = {}

    def __new__(mcs, name, bases, attrs):
        cls = super(CurrencyGetterType, mcs).__new__(mcs, name, bases, attrs)
        if getattr(cls, 'code', None):
            mcs.getters[cls.code] = cls
        return cls

    @classmethod
    def get(mcs, code, *args, **kwargs):
        """ Get getter by code
        """
        return mcs.getters[code](*args, **kwargs)


class CurrencyGetterInterface(object):
    """ Abstract class of currency getter

        To create new getter, just subclass this class
        and define class variables 'code' and 'name'
        and implement *get_updated_currency* method

        For example::

            from odoo.addons.currency_rate_update \
                import CurrencyGetterInterface

            class MySuperCurrencyGetter(CurrencyGetterInterface):
                code = "MSCG"
                name = "My Currency Rates"
                supported_currency_array = ['USD', 'EUR']

                def get_updated_currency(self, currency_array, main_currency,
                                         max_delta_days):
                    # your code that fills self.updated_currency

                    # and return result
                    return self.updated_currency, self.log_info

    """
    __metaclass__ = CurrencyGetterType

    # attributes required for currency getters
    code = None  # code for service selection
    name = None  # displayed name

    log_info = " "

    supported_currency_array = [
        'AED', 'AFN', 'ALL', 'AMD', 'ANG', 'AOA', 'ARS', 'AUD', 'AWG', 'AZN',
        'BAM', 'BBD', 'BDT', 'BGN', 'BHD', 'BIF', 'BMD', 'BND', 'BOB', 'BRL',
        'BSD', 'BTN', 'BWP', 'BYR', 'BZD', 'CAD', 'CDF', 'CHF', 'CLP', 'CNY',
        'COP', 'CRC', 'CUP', 'CVE', 'CYP', 'CZK', 'DJF', 'DKK', 'DOP', 'DZD',
        'EEK', 'EGP', 'ERN', 'ETB', 'EUR', 'FJD', 'FKP', 'GBP', 'GEL', 'GGP',
        'GHS', 'GIP', 'GMD', 'GNF', 'GTQ', 'GYD', 'HKD', 'HNL', 'HRK', 'HTG',
        'HUF', 'IDR', 'ILS', 'IMP', 'INR', 'IQD', 'IRR', 'ISK', 'JEP', 'JMD',
        'JOD', 'JPY', 'KES', 'KGS', 'KHR', 'KMF', 'KPW', 'KRW', 'KWD', 'KYD',
        'KZT', 'LAK', 'LBP', 'LKR', 'LRD', 'LSL', 'LTL', 'LVL', 'LYD', 'MAD',
        'MDL', 'MGA', 'MKD', 'MMK', 'MNT', 'MOP', 'MRO', 'MTL', 'MUR', 'MVR',
        'MWK', 'MXN', 'MYR', 'MZN', 'NAD', 'NGN', 'NIO', 'NOK', 'NPR', 'NZD',
        'OMR', 'PAB', 'PEN', 'PGK', 'PHP', 'PKR', 'PLN', 'PYG', 'QAR', 'RON',
        'RSD', 'RUB', 'RWF', 'SAR', 'SBD', 'SCR', 'SDG', 'SEK', 'SGD', 'SHP',
        'SLL', 'SOS', 'SPL', 'SRD', 'STD', 'SVC', 'SYP', 'SZL', 'THB', 'TJS',
        'TMM', 'TND', 'TOP', 'TRY', 'TTD', 'TVD', 'TWD', 'TZS', 'UAH', 'UGX',
        'USD', 'UYU', 'UZS', 'VEB', 'VEF', 'VND', 'VUV', 'WST', 'XAF', 'XAG',
        'XAU', 'XCD', 'XDR', 'XOF', 'XPD', 'XPF', 'XPT', 'YER', 'ZAR', 'ZMK',
        'ZWD'
    ]

    # Updated currency this arry will contain the final result
    updated_currency = {}

    def get_updated_currency(self, currency_array, main_currency,
                             max_delta_days):
        """Interface method that will retrieve the currency
           This function has to be reinplemented in child
        """
        raise AbstractMethodError

    def validate_cur(self, currency):
        """Validate if the currency to update is supported"""
        if currency not in self.supported_currency_array:
            raise UnsuportedCurrencyError(currency)

    def get_url(self, url):
        """Return a string of a get url query"""
        try:
            import urllib
            objfile = urllib.urlopen(url)
            rawfile = objfile.read()
            objfile.close()
            return rawfile
        except ImportError:
            raise UserError(
                _('Unable to import urllib.'))
        except IOError:
            raise UserError(
                _('Web Service does not exist (%s)!') % url)

    def check_rate_date(self, rate_date, max_delta_days):
        """Check date constrains. rate_date must be of datetime type"""
        days_delta = (datetime.today() - rate_date).days
        if days_delta > max_delta_days:
            raise Exception(
                'The rate timestamp %s is %d days away from today, '
                'which is over the limit (%d days). '
                'Rate not updated in Odoo.' % (rate_date,
                                               days_delta,
                                               max_delta_days)
            )

        # We always have a warning when rate_date != today
        if rate_date.date() != datetime.today().date():
            rate_date_str = fields.Date.to_string(rate_date)
            msg = "The rate timestamp %s is not today's date %s" % \
                (rate_date_str, fields.Date.today())
            self.log_info = ("\n WARNING : %s") % msg
            _logger.warning(msg)
