# Copyright 2022 Camptocamp SA (https://www.camptocamp.com).
# @author Iván Todorovich <ivan.todorovich@camptocamp.com>
# License AGPL-3.0 or later (https://www.gnu.org/licenses/agpl).

from odoo import Command
from odoo.tests import TransactionCase


class TestMailLayoutForce(TransactionCase):
    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls.layout_noop = cls.env.ref("mail_layout_force.mail_layout_noop")
        cls.layout_test = cls.env["ir.ui.view"].create(
            {
                "name": "Test Layout",
                "type": "qweb",
                "mode": "primary",
                "arch": "<t t-name='test'><h1></h1><t t-out='message.body'/></t>",
            }
        )
        cls.mail_notification_layout = cls.env.ref("mail.mail_notification_layout")
        cls.layout_substitute = cls.env["ir.ui.view"].create(
            {
                "name": "Substitute Layout",
                "type": "qweb",
                "mode": "primary",
                "arch": """<?xml version="1.0"?>
                    <t t-name="custom_mail_notification_layout">
                    <div t-out="message.body"/>
                    <div t-if="signature" t-out="signature" style="font-size: 13px;"/>
                    <h1>Substituted</h1>
                </t>""",
            }
        )
        cls.template = cls.env["mail.template"].create(
            {
                "name": "Test Template",
                "body_html": "<p>Test</p>",
                "subject": "Test",
                "model_id": cls.env.ref("base.model_res_partner").id,
                "auto_delete": False,
            }
        )
        cls.partner = cls.env.ref("base.res_partner_10")
        cls.partner.message_ids.unlink()
        cls.partner.message_subscribe([cls.partner.id])

    def test_noop_layout(self):
        self.template.force_email_layout_id = self.layout_noop
        self.partner.message_post_with_template(
            self.template.id,
            # This is ignored because the template has a force_email_layout_id
            email_layout_xmlid="mail.mail_notification_light",
        )
        message = self.partner.message_ids[-1]
        self.assertEqual(message.mail_ids.body_html.strip(), "<p>Test</p>")

    def test_custom_layout(self):
        self.template.force_email_layout_id = self.layout_test
        self.partner.message_post_with_template(
            self.template.id,
            # This is ignored because the template has a force_email_layout_id
            email_layout_xmlid="mail.mail_notification_light",
        )
        message = self.partner.message_ids[-1]
        self.assertEqual(message.mail_ids.body_html.strip(), "<h1></h1><p>Test</p>")

    def test_custom_layout_composer(self):
        self.template.force_email_layout_id = self.layout_test
        composer = (
            self.env["mail.compose.message"]
            .with_context(
                # This is ignored because the template has a force_email_layout_id
                custom_layout="mail.mail_notification_light"
            )
            .create(
                {
                    "res_id": self.partner.id,
                    "model": self.partner._name,
                    "template_id": self.template.id,
                }
            )
        )
        composer._onchange_template_id_wrapper()
        composer._action_send_mail()
        message = self.partner.message_ids[-1]
        self.assertEqual(message.mail_ids.body_html.strip(), "<h1></h1><p>Test</p>")

    def test_chatter_message_uses_default_layout(self):
        self.partner.message_post(
            body="Test Message",
            email_layout_xmlid=self.mail_notification_layout.xml_id,
            message_type="comment",
            subtype_xmlid="mail.mt_comment",
            mail_auto_delete=False,
            force_send=True,
        )
        message = self.partner.message_ids[-1]
        self.assertNotIn("<h1>Substituted</h1>", message.mail_ids.body_html)
        self.assertIn("Test Message", message.mail_ids.body_html)

    def test_chatter_message_uses_substituted_layout(self):
        self.mail_notification_layout.layout_mapping_line_ids = [
            Command.create({"substitute_layout_id": self.layout_substitute.id})
        ]
        self.partner.message_post(
            body="Test Message",
            email_layout_xmlid=self.mail_notification_layout.xml_id,
            message_type="comment",
            subtype_xmlid="mail.mt_comment",
            mail_auto_delete=False,
            force_send=True,
        )
        message = self.partner.message_ids[-1]
        self.assertIn("<h1>Substituted</h1>", message.mail_ids.body_html)
        self.assertIn("Test Message", message.mail_ids.body_html)
