# For copyright and license notices, see __manifest__.py file in module root
# directory or check the readme files

import hashlib
import logging
import os
import sys
import time
import traceback

import odoo.tools as tools
from odoo import _, api, fields, models
from odoo.exceptions import UserError

_logger = logging.getLogger(__name__)


class ResCompany(models.Model):

    _inherit = "res.company"

    afip_ws_env_type = fields.Selection(
        [("homologation", "Homologation"), ("production", "Production")],
        string="AFIP WS Environment",
        default="production",
        help="Environment is used to connect AFIP Web Services.\n"
        "Production: This is the connection used in real world.\n"
        "Homologation: Use this environment to test AFIP Web Services.",
    )

    alias_ids = fields.One2many(
        "afipws.certificate_alias",
        "company_id",
        "Aliases",
        auto_join=True,
    )
    connection_ids = fields.One2many(
        "afipws.connection",
        "company_id",
        "Connections",
        auto_join=True,
    )

    def get_key_and_certificate(self, environment_type):
        """
        Funcion que busca para el environment_type definido,
        una clave y un certificado en los siguientes lugares y segun estas
        prioridades:
        * en el conf del server de odoo
        * en registros de esta misma clase
        """
        self.ensure_one()
        pkey = False
        cert = False
        msg = False
        certificate = self.env["afipws.certificate"].search(
            [
                ("alias_id.company_id", "=", self.id),
                ("alias_id.type", "=", environment_type),
                ("state", "=", "confirmed"),
            ]
        )
        # to avoid confusion on the user, if more than one certificate found,
        # we ask to keep the one he whants to use
        if len(certificate) > 1:
            raise UserError(
                _(
                    'Tiene más de un certificado de "%(environment_type)s" confirmado.'
                    'Por favor, deje un solo certificado de "%(environment_type)s" confirmado.'
                )
                % {"environment_type": environment_type}
            )
        if certificate:
            pkey = certificate.alias_id.key
            cert = certificate.crt
            _logger.info("Using DB certificates")
        # not certificate on bd, we search on odo conf file
        else:
            msg = _(
                'Not confirmed certificate for "%(environment_type)s" on company "%(name)s"'
            ) % {"environment_type": environment_type, "name": self.name}
            pkey_path = False
            cert_path = False
            if environment_type == "production":
                pkey_path = tools.config.get("afip_prod_pkey_file")
                cert_path = tools.config.get("afip_prod_cert_file")
            else:
                pkey_path = tools.config.get("afip_homo_pkey_file")
                cert_path = tools.config.get("afip_homo_cert_file")
            if pkey_path and cert_path:
                try:
                    if os.path.isfile(pkey_path) and os.path.isfile(cert_path):
                        with open(pkey_path, "r") as pkey_file:
                            pkey = pkey_file.read()
                        with open(cert_path, "r") as cert_file:
                            cert = cert_file.read()
                    msg = "Could not find %s or %s files" % (pkey_path, cert_path)
                except Exception:
                    msg = "Could not read %s or %s files" % (pkey_path, cert_path)
                else:
                    _logger.info("Using odoo conf certificates")
        if not pkey or not cert:
            raise UserError(msg)
        return (pkey, cert)

    def get_connection(self, afip_ws):
        self.ensure_one()
        _logger.info(
            "Getting connection for company %s and ws %s" % (self.name, afip_ws)
        )
        now = fields.Datetime.now()
        environment_type = self.afip_ws_env_type

        connection = self.connection_ids.search(
            [
                ("type", "=", environment_type),
                ("generationtime", "<=", now),
                ("expirationtime", ">", now),
                ("afip_ws", "=", afip_ws),
                ("company_id", "=", self.id),
            ],
            limit=1,
        )
        if not connection:
            connection = self._create_connection(afip_ws, environment_type)
        return connection

    def _create_connection(self, afip_ws, environment_type):
        """
        This function should be called from get_connection. Not to be used
        directyl
        TODO ver si podemos usar metodos de pyafipws para esto
        """
        self.ensure_one()
        _logger.info(
            "Creating connection for company %s, environment type %s and ws "
            "%s" % (self.name, environment_type, afip_ws)
        )
        login_url = self.env["afipws.connection"].get_afip_login_url(environment_type)
        pkey, cert = self.get_key_and_certificate(environment_type)
        # because pyafipws wsaa loos for "BEGIN RSA PRIVATE KEY" we change key
        if pkey.startswith("-----BEGIN PRIVATE KEY-----"):
            pkey = pkey.replace(" PRIVATE KEY", " RSA PRIVATE KEY")
        auth_data = self.authenticate(afip_ws, cert, pkey, wsdl=login_url)
        auth_data.update(
            {
                "company_id": self.id,
                "afip_ws": afip_ws,
                "type": environment_type,
            }
        )
        _logger.info("Successful Connection to AFIP.")
        generationtime = auth_data["generationtime"].replace("T", " ")
        auth_data["generationtime"] = generationtime[:19]
        expirationtime = auth_data["expirationtime"].replace("T", " ")
        auth_data["expirationtime"] = expirationtime[:19]
        return self.connection_ids.create(auth_data)

    @api.model
    def authenticate(
        self,
        service,
        certificate,
        private_key,
        force=False,
        cache="",
        wsdl="",
        proxy="",
    ):
        """
        Call AFIP Authentication webservice to get token & sign or error
        message
        """
        # import AFIP webservice authentication helper:
        from pyafipws.wsaa import WSAA

        # create AFIP webservice authentication helper instance:
        wsaa = WSAA()
        # raise python exceptions on any failure
        wsaa.LanzarExcepciones = True

        # five hours
        DEFAULT_TTL = 60 * 60 * 5

        # make md5 hash of the parameter for caching...
        fn = (
            "%s.xml"
            % hashlib.md5(
                (service + certificate + private_key).encode("utf-8")
            ).hexdigest()
        )
        if cache:
            fn = os.path.join(cache, fn)
        else:
            fn = os.path.join(wsaa.InstallDir, "cache", fn)

        try:
            # read the access ticket (if already authenticated)
            if (
                not os.path.exists(fn)
                or os.path.getmtime(fn) + (DEFAULT_TTL) < time.time()
            ):
                # access ticket (TA) outdated, create new access request
                # ticket (TRA)
                tra = wsaa.CreateTRA(service=service, ttl=DEFAULT_TTL)
                # cryptographically sing the access ticket
                cms = wsaa.SignTRA(tra, certificate, private_key)
                # connect to the webservice:
                wsaa.Conectar(cache, wsdl, proxy)
                # call the remote method
                ta = wsaa.LoginCMS(cms)
                if not ta:
                    raise RuntimeError()
                # write the access ticket for further consumption
                open(fn, "w").write(ta)
            else:
                # get the access ticket from the previously written file
                ta = open(fn, "r").read()
            # analyze the access ticket xml and extract the relevant fields
            wsaa.AnalizarXml(xml=ta)
            token = wsaa.ObtenerTagXml("token")
            sign = wsaa.ObtenerTagXml("sign")
            expirationTime = wsaa.ObtenerTagXml("expirationTime")
            generationTime = wsaa.ObtenerTagXml("generationTime")
            uniqueId = wsaa.ObtenerTagXml("uniqueId")
        except Exception:
            token = sign = None
            if wsaa.Excepcion:
                # get the exception already parsed by the helper
                err_msg = wsaa.Excepcion
            else:
                # avoid encoding problem when reporting exceptions to the user:
                err_msg = traceback.format_exception_only(sys.exc_type, sys.exc_value)[
                    0
                ]
            raise UserError(
                _("Could not connect. This is the what we received: %s") % (err_msg)
            ) from Exception
        return {
            "uniqueid": uniqueId,
            "generationtime": generationTime,
            "expirationtime": expirationTime,
            "token": token,
            "sign": sign,
        }
