# -*- coding: utf-8 -*-
# © 2015 Oihane Crucelaegui
# © 2015-2016 Pedro M. Baeza <pedro.baeza@tecnativa.com>
# License AGPL-3 - See http://www.gnu.org/licenses/agpl-3.0.html

from openerp.tests import common
from openerp.exceptions import AccessError


@common.at_install(False)
@common.post_install(True)
class TestPartnerMultiCompany(common.SavepointCase):
    @classmethod
    def setUpClass(cls):
        super(TestPartnerMultiCompany, cls).setUpClass()
        # This is for being sure that the suspend_security method is hooked
        cls.env['ir.rule']._register_hook()
        # Avoid possible spam
        cls.partner_model = cls.env['res.partner'].with_context(
            mail_create_nosubscribe=True,
        )
        cls.company_1 = cls.env['res.company'].create(
            {'name': 'Test company 1'})
        cls.company_2 = cls.env['res.company'].create(
            {'name': 'Test company 2'})
        cls.partner_company_none = cls.partner_model.create(
            {'name': 'partner without company',
             'company_ids': False})
        cls.partner_company_1 = cls.partner_model.create(
            {'name': 'partner from company 1',
             'company_ids': [(6, 0, cls.company_1.ids)]})
        cls.partner_company_2 = cls.partner_model.create(
            {'name': 'partner from company 2',
             'company_ids': [(6, 0, cls.company_2.ids)]})
        cls.partner_company_both = cls.partner_model.create(
            {'name': 'partner for both companies',
             'company_ids': [(6, 0, (cls.company_1 + cls.company_2).ids)]})
        cls.user_company_1 = cls.env['res.users'].create(
            {'name': 'User company 1',
             'login': 'user_company_1',
             'email': 'somebody@somewhere.com',
             'groups_id': [
                 (6, 0, cls.env.ref('base.group_partner_manager').ids)],
             'company_id': cls.company_1.id,
             'company_ids': [(6, 0, cls.company_1.ids)]})
        cls.user_company_2 = cls.env['res.users'].create(
            {'name': 'User company 2',
             'login': 'user_company_2',
             'email': 'somebody@somewhere.com',
             'groups_id': [
                 (6, 0, cls.env.ref('base.group_partner_manager').ids)],
             'company_id': cls.company_2.id,
             'company_ids': [(6, 0, cls.company_2.ids)]})

    def test_create_partner(self):
        partner = self.env['res.partner'].create({'name': 'Test'})
        company = self.env['res.company']._company_default_get('res.partner')
        self.assertIn(company, partner.company_ids)
        partner = self.env['res.partner'].create({
            'name': 'Test 2',
            'company_ids': [(4, self.company_1.id)],
        })
        self.assertEqual(partner.company_id, self.company_1)
        partner = self.env['res.partner'].create({
            'name': 'Test 2',
            'company_ids': [(5, False)],
        })
        self.assertFalse(partner.company_id)

    def test_company_none(self):
        self.assertFalse(self.partner_company_none.company_id)
        # All of this should be allowed
        self.partner_company_none.sudo(self.user_company_1.id).name = "Test"
        self.partner_company_none.sudo(self.user_company_2.id).name = "Test"

    def test_company_1(self):
        self.assertEqual(self.partner_company_1.company_id, self.company_1)
        # All of this should be allowed
        self.partner_company_1.sudo(self.user_company_1).name = "Test"
        self.partner_company_both.sudo(self.user_company_1).name = "Test"
        # And this one not
        with self.assertRaises(AccessError):
            self.partner_company_2.sudo(self.user_company_1).name = "Test"

    def test_create_company_1(self):
        partner = self.partner_model.sudo(self.user_company_1).create(
            {'name': 'Test from user company 1'})
        self.assertEqual(partner.company_id, self.company_1)

    def test_create_company_2(self):
        partner = self.partner_model.sudo(self.user_company_2).create(
            {'name': 'Test from user company 2'})
        self.assertEqual(partner.company_id, self.company_2)

    def test_company_2(self):
        self.assertEqual(self.partner_company_2.company_id, self.company_2)
        # All of this should be allowed
        self.partner_company_2.sudo(self.user_company_2).name = "Test"
        self.partner_company_both.sudo(self.user_company_2).name = "Test"
        # And this one not
        with self.assertRaises(AccessError):
            self.partner_company_1.sudo(self.user_company_2).name = "Test"

    def test_uninstall(self):
        from ..hooks import uninstall_hook
        uninstall_hook(self.env.cr, None)
        rule = self.env.ref('base.res_partner_rule')
        domain = ("['|','|',"
                  "('company_id.child_ids','child_of',[user.company_id.id]),"
                  "('company_id','child_of',[user.company_id.id]),"
                  "('company_id','=',False)]")
        self.assertEqual(rule.domain_force, domain)
        self.assertFalse(rule.active)

    def test_switch_user_company(self):
        self.user_company_1.company_ids = (self.company_1 + self.company_2).ids
        self.user_company_1.company_id = self.company_2.id
        self.assertEqual(
            self.user_company_1.partner_id.company_id, self.company_2)

    def test_init_hook(self):
        """It should set company_ids even on deactivated partner."""
        deactivated_partner = self.env.ref('base.public_user_res_partner')
        self.assertFalse(deactivated_partner.active)
        self.assertEqual(deactivated_partner.company_ids.ids,
                         deactivated_partner.company_id.ids)
