# Copyright (C) 2022 NextERP Romania
# Copyright (C) 2020 Terrabit
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html).

import logging

from odoo import models
from odoo.tools.float_utils import float_is_zero

_logger = logging.getLogger(__name__)


class AccountMoveLine(models.Model):
    _name = "account.move.line"
    _inherit = ["account.move.line", "l10n.ro.mixin"]

    def _apply_price_difference(self):
        l10n_ro_records = self.filtered("is_l10n_ro_record")
        res = super(AccountMoveLine, self - l10n_ro_records)._apply_price_difference()
        return res

    def _generate_price_difference_vals(self, layers):
        l10n_ro_records = self.filtered("is_l10n_ro_record")
        svl_vals_list, aml_vals_list = super(
            AccountMoveLine, self - l10n_ro_records
        )._generate_price_difference_vals(layers)
        return svl_vals_list, aml_vals_list

    def l10n_ro_get_stock_valuation_difference(self):
        """Se obtine diferenta dintre valoarea stocului si valoarea din factura"""
        line = self
        diff, qty_diff = 0.0, 0.0
        # Retrieve stock valuation moves.
        if not line.purchase_line_id:
            return diff, qty_diff

        if line.purchase_line_id.product_id.purchase_method != "receive":
            return diff, qty_diff

        valuation_stock_moves = self._l10n_ro_get_valuation_stock_moves()
        if not valuation_stock_moves:
            return diff, qty_diff

        valuation_total = 0
        valuation_total_qty = 0
        for val_stock_move in valuation_stock_moves:
            # Take normal svl and svl generated by price difference.
            svl = (
                val_stock_move.sudo()
                .mapped("stock_valuation_layer_ids")
                .filtered(
                    lambda l: not l.stock_landed_cost_id
                    or l.stock_landed_cost_id.l10n_ro_cost_type == "price_diff"
                )
            )
            layers_qty = sum(svl.mapped("quantity"))
            layers_values = sum(svl.mapped("value"))

            valuation_total += layers_values
            valuation_total_qty += layers_qty

        precision = line.product_uom_id.rounding or line.product_id.uom_id.rounding
        if float_is_zero(valuation_total_qty, precision_rounding=precision):
            return diff, qty_diff

        lines = self.search(
            [
                ("purchase_line_id", "=", line.purchase_line_id.id),
                ("move_id.state", "!=", "cancel"),
            ]
        )
        inv_qty = 0
        for line in lines:
            line_qty = line.product_uom_id._compute_quantity(
                line.quantity, line.product_id.uom_id
            )
            inv_qty += (-1 if line.move_id.move_type == "in_refund" else 1) * line_qty

        if inv_qty * valuation_total_qty:
            accc_balance = sum(lines.mapped("balance")) / inv_qty * valuation_total_qty
        else:
            accc_balance = 0
        diff = abs(accc_balance) - valuation_total
        currency = line.currency_id or self.env.company.currency_id
        diff = currency.round(diff)
        qty_diff = inv_qty - valuation_total_qty
        return diff, qty_diff

    def l10n_ro_modify_stock_valuation(self, val_dif):
        # se adauga la evaluarea miscarii de stoc
        self.ensure_one()
        if not self.purchase_line_id:
            return 0.0
        valuation_stock_move = self.env["stock.move"].search(
            [
                ("purchase_line_id", "=", self.purchase_line_id.id),
                ("state", "=", "done"),
                ("product_qty", "!=", 0.0),
            ],
            order="id desc",
            limit=1,
        )
        value = val_dif
        # trebuie cantitate din factura in unitatea produsului si apoi
        # value = self.product_uom_id._compute_price(value, self.product_id.uom_id)

        lc = self._l10n_ro_create_price_difference_landed_cost(value)
        lc.compute_landed_cost()
        lc.with_context(
            l10n_ro_price_difference_move_ids=valuation_stock_move
        ).button_validate()

        lc.stock_valuation_layer_ids.mapped("account_move_id")

        svl = lc.stock_valuation_layer_ids.filtered(
            lambda svl: svl.value == lc.amount_total
        )

        svl.write(
            {
                "quantity": 0,
                "remaining_qty": 0,
                "description": "Price Difference",
                "l10n_ro_invoice_line_id": self.id,
                "l10n_ro_invoice_id": self.move_id.id,
            }
        )

    def l10n_ro_prepare_price_difference_landed_cost(self, value):
        price_diff_product = self._l10n_ro_get_or_create_price_difference_product()
        stock_journal_id = self.product_id.categ_id.property_stock_journal or False
        return dict(
            account_journal_id=stock_journal_id and stock_journal_id.id,
            l10n_ro_cost_type="price_diff",
            cost_lines=[
                (
                    0,
                    0,
                    {
                        "name": "Price Difference: equal split",
                        "split_method": "by_quantity",
                        "price_unit": value,
                        "product_id": price_diff_product.id,
                        "account_id": self.account_id.id,
                    },
                )
            ],
        )

    def _l10n_ro_create_price_difference_landed_cost(self, value):
        vals = self.l10n_ro_prepare_price_difference_landed_cost(value)
        return self.env["stock.landed.cost"].sudo().create(vals)

    def _l10n_ro_get_or_create_price_difference_product(self):
        price_diff_product = (
            self.company_id.l10n_ro_property_stock_price_difference_product_id
        )
        if not price_diff_product:
            serv_acc = self.env["account.account"].search(
                [
                    ("account_type", "=", "expense"),
                    ("company_id", "=", self.company_id.id),
                ],
                limit=1,
            )

            price_diff_product = self.env["product.product"].create(
                {
                    "name": "Price Difference Between Reception and Bill",
                    "categ_id": self.env.ref("product.product_category_all").id,
                    "type": "service",
                    "landed_cost_ok": True,
                    "property_account_expense_id": serv_acc if serv_acc else False,
                }
            )

            self.sudo().company_id.l10n_ro_property_stock_price_difference_product_id = (
                price_diff_product
            )

        return price_diff_product
