As specified, this module will not add any new features by itself once installed. As a
developer, you can depend on this module to benefit from a set of specialized methods
that can be used to get and generate thumbnails from an image you want to index.

Behind the scenes, this module extend the `se.indexable.recor` model by adding 2 new
methods:

- `_get_or_create_thumbnails_for_multi_images`: This method is used to get or create
  thumbnails for a set of images.
- `_get_or_create_thumbnails_for_image`: This method is used to get or create thumbnails
  for a single image.

Thanks to these methods, you can easily generate thumbnails for images stored into odoo
through three different ways:

- A relational field referencing record inheriting from `fs.image.relation.mixin`.
- An odoo's Image field.
- An FSImage field.

The sizes of the thumbnails generated by these methods are configured through the
creation of a new record into the `se.thumbnail.size` model. This model allows you to
define the size of thumbnails by search engine backend, model and field. For example,
you can define that you want to generate thumbnails of size `100x100` for all the images
stored into the `product.template` model and the `image` field when you export them to
the `opensearch_xyz` backend. In this way, you can easily define different thumbnail
sizes for different search engine backends, models and fields according to your needs.

Thumbnails are always retrieved and generated for one image and a specific `base_name`.
The base name is used as a key to identify the thumbnails from an image related to a
usage and also as base name for the file name of the thumbnails generated. By default,
the `base_name` is computed from the display name of the record that contains the image
to ensure a good SEO. To avoid problems with special characters, the `base_name` is
normalized before being used.

Here is an example of how to use these methods:

```python

from odoo import models, fields
from odoo.addons.fs_imeage.fields import FSImage


class ProductImageRelation(models.Model):
    _name = "product.image.relation"
    _inherit = ["fs.image.relation.mixin"]

    name = fields.Char(required=True)
    product_id = fields.Many2one(
        "product.product",
        required=True,
        ondelete="cascade",
    )


class ProductProduct(models.Model):
    _name = "product.product"
    _inherit = ["se.indexable.record", "product.product"]

    name = fields.Char(required=True)
    image = fields.Image(required=True)
    image_ids = fields.One2many(
        "product.image.relation",
        "product_id",
        string="Images",
    )
    fs_image = FSImage(required=True)



# A creation is always done for a given se.index record.
index = self.env["se.index"].browse(1)
product = self.env["product.product"].browse(1)

# Get or create thumbnails for a single image
for (thumbnail_size, thumbnail) in product._get_or_create_thumbnails_for_image(
    index,
    field_name="image",
    ):
    # Do something with the thumbnail
    print(f"Thumbnail  for image with size {thumbnail_size.display_name} "
        "is available at url {thumbnail.image.url}")

# Get or create thumbnails for a single image
for (thumbnail_size, thumbnail) in product._get_or_create_thumbnails_for_image(
    index,
    field_name="fs_image",
    ):
    # Do something with the thumbnail
    print(f"Thumbnail for fs_image with size {thumbnail_size.display_name} "
        "is available at url {thumbnail.image.url}")


# Get or create thumbnails for a set of images
thumbnails_by_image = product._get_or_create_thumbnails_for_multi_images(
    index,
    field_name="image_ids",
    )

for (image, thumbnails) in thumbnails_by_image.items():
    for (thumbnail_size, thumbnail) in thumbnails:
        # Do something with the thumbnail
        print(f"Thumbnail for image {image.name} with size "
            "{thumbnail_size.display_name} is available at url "
            "{thumbnail.image.url}")
```
