# Copyright 2019 ForgeFlow S.L.
# Copyright 2019 Serpent Consulting Services Pvt. Ltd.
# License LGPL-3.0 or later (https://www.gnu.org/licenses/lgpl.html).
from odoo import _, api, fields, models
from odoo.exceptions import UserError


class StockLocation(models.Model):
    _inherit = "stock.location"

    operating_unit_id = fields.Many2one(
        comodel_name="operating.unit",
        string="Operating Unit",
        check_company=True,
    )

    @api.constrains("operating_unit_id")
    def _check_warehouse_operating_unit(self):
        for rec in self:
            warehouse_obj = self.env["stock.warehouse"]
            warehouses = warehouse_obj.search(
                [
                    "|",
                    "|",
                    ("wh_input_stock_loc_id", "=", rec.ids[0]),
                    ("lot_stock_id", "in", rec.ids),
                    ("wh_output_stock_loc_id", "in", rec.ids),
                ]
            )
            for w in warehouses:
                if w.operating_unit_id and rec.operating_unit_id != w.operating_unit_id:
                    raise UserError(
                        _(
                            "Configuration error. This location is "
                            "assigned to a warehouse that belongs to"
                            " a different operating unit."
                        )
                    )

    @api.constrains("operating_unit_id")
    def _check_required_operating_unit(self):
        for rec in self:
            if rec.usage in ("supplier", "customer") and rec.operating_unit_id:
                raise UserError(
                    _(
                        "Configuration error. The operating unit should be "
                        "assigned to internal locations only."
                    )
                )

    @api.constrains("operating_unit_id", "location_id")
    def _check_parent_operating_unit(self):
        for rec in self:
            if (
                rec.location_id
                and rec.location_id.usage == "internal"
                and rec.operating_unit_id
                and rec.operating_unit_id != rec.location_id.operating_unit_id
            ):
                raise UserError(
                    _(
                        "Configuration error. The Parent Stock Location "
                        "must belong to the same Operating Unit."
                    )
                )
