# -*- coding: utf-8 -*-
# Copyright 2017 FactorLibre - Ismael Calvo <ismael.calvo@factorlibre.com>
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html)

from openerp.tests import common
from openerp import exceptions, fields


def _deep_sort(obj):
    """
    Recursively sort list or dict nested lists
    """
    if isinstance(obj, dict):
        _sorted = {}
        for key in sorted(obj):
            _sorted[key] = _deep_sort(obj[key])
    elif isinstance(obj, list):
        new_list = []
        for val in obj:
            new_list.append(_deep_sort(val))
        _sorted = sorted(new_list)
    else:
        _sorted = obj
    return _sorted


class TestL10nEsAeatSii(common.TransactionCase):
    def setUp(self):
        super(TestL10nEsAeatSii, self).setUp()
        self.maxDiff = None
        self.partner = self.env['res.partner'].create({
            'name': 'Test partner',
            'vat': 'ESF35999705'
        })
        self.product = self.env['product.product'].create({
            'name': 'Test product',
        })
        self.account_type = self.env['account.account.type'].create({
            'name': 'Test account type',
            'code': 'TEST',
        })
        self.account_expense = self.env['account.account'].create({
            'name': 'Test expense account',
            'code': 'EXP',
            'type': 'other',
            'user_type': self.account_type.id,
        })
        self.analytic_account = self.env['account.analytic.account'].create({
            'name': 'Test analytic account',
            'type': 'normal',
        })
        self.account_tax = self.env['account.account'].create({
            'name': 'Test tax account',
            'code': 'TAX',
            'type': 'other',
            'user_type': self.account_type.id,
        })
        self.base_code = self.env['account.tax.code'].create({
            'name': '[28] Test base code',
            'code': 'OICBI',
        })
        self.tax_code = self.env['account.tax.code'].create({
            'name': '[29] Test tax code',
            'code': 'SOICC',
        })
        self.tax = self.env['account.tax'].create({
            'name': 'Test tax 10%',
            # Needed for discriminatory tax amount in supplier invoices
            'description': 'P_IVA10_BC',
            'type_tax_use': 'purchase',
            'type': 'percent',
            'amount': '0.10',
            'account_collected_id': self.account_tax.id,
            'base_code_id': self.base_code.id,
            'base_sign': 1,
            'tax_code_id': self.tax_code.id,
            'tax_sign': 1,
        })
        self.tax_irpf19 = self.env['account.tax'].create({
            'name': 'Test IRPF 19%',
            # Needed for discriminatory tax amount in supplier invoices
            'description': 'P_IRPF19',
            'type_tax_use': 'purchase',
            'type': 'percent',
            'amount': '0.19',
            'account_collected_id': self.account_tax.id,
            'base_code_id': self.base_code.id,
            'base_sign': 1,
            'tax_code_id': self.tax_code.id,
            'tax_sign': 1,
        })
        self.period = self.env['account.period'].find()
        self.env.user.company_id.sii_description_method = 'manual'
        self.invoice = self.env['account.invoice'].create({
            'partner_id': self.partner.id,
            'date_invoice': fields.Date.today(),
            'type': 'out_invoice',
            'period_id': self.period.id,
            'account_id': self.partner.property_account_payable.id,
            'invoice_line': [
                (0, 0, {
                    'product_id': self.product.id,
                    'account_id': self.account_expense.id,
                    'account_analytic_id': self.analytic_account.id,
                    'name': 'Test line',
                    'price_unit': 100,
                    'quantity': 1,
                    'invoice_line_tax_id': [(6, 0, self.tax.ids)],
                })],
            'sii_manual_description': '/',
        })
        self.user = self.env['res.users'].create({
            'name': 'Test user',
            'login': 'test_user',
            'groups_id': [
                (4, self.env.ref('account.group_account_manager').id),
                (4, self.env.ref('l10n_es_aeat.group_account_aeat').id)
            ],
            'email': 'somebody@somewhere.com',
        })

    def _open_invoice(self):
        self.invoice.company_id.write({
            'sii_enabled': True,
            'sii_test': True,
            'use_connector': True,
            'chart_template_id': self.env.ref(
                'l10n_es.account_chart_template_pymes').id,
            'vat': 'ESU2687761C',
        })
        self.invoice.signal_workflow('invoice_open')

    def test_job_creation(self):
        self._open_invoice()
        self.assertTrue(self.invoice.invoice_jobs_ids)

    def _get_invoices_test(self, invoice_type, special_regime):
        str_today = self.invoice._change_date_format(fields.Date.today())
        emisor = self.invoice.company_id
        contraparte = self.partner
        expedida_recibida = 'FacturaExpedida'
        if self.invoice.type in ['in_invoice', 'in_refund']:
            emisor = self.partner
            expedida_recibida = 'FacturaRecibida'
        res = {
            'IDFactura': {
                'FechaExpedicionFacturaEmisor': str_today,
                'IDEmisorFactura': {
                    'NIF': emisor.vat[2:]},
                'NumSerieFacturaEmisor': (
                    self.invoice.supplier_invoice_number or
                    self.invoice.number)},
            expedida_recibida: {
                'TipoFactura': invoice_type,
                'Contraparte': {
                    'NombreRazon': contraparte.name,
                    'NIF': contraparte.vat[2:],
                },
                'DescripcionOperacion': u'/',
                'ClaveRegimenEspecialOTrascendencia': special_regime,
                'ImporteTotal': self.invoice.cc_amount_total,
            },
            'PeriodoLiquidacion': {
                'Periodo': str(self.invoice.period_id.code[:2]),
                'Ejercicio': int(self.invoice.period_id.code[-4:])
            }
        }
        if self.invoice.type in ['out_invoice', 'out_refund']:
            res[expedida_recibida].update({
                'TipoDesglose': {},
            })
        else:
            res[expedida_recibida].update({
                "FechaRegContable": self.invoice._change_date_format(
                    self.invoice.date_invoice
                ),
                "DesgloseFactura": {
                    'DesgloseIVA': {
                        'DetalleIVA': [
                            {
                                'BaseImponible': 100.0,
                                'CuotaRepercutida': 0,
                                'CuotaSoportada': 10.0,
                                'TipoImpositivo': '10.0',
                            },
                        ],
                    },
                },
                "CuotaDeducible": self.invoice.cc_amount_tax,
            })
        if invoice_type == 'R4':
            invoices = self.invoice.origin_invoices_ids
            base_rectificada = sum(invoices.mapped('cc_amount_untaxed'))
            cuota_rectificada = sum(invoices.mapped('cc_amount_tax'))
            res[expedida_recibida].update({
                'TipoRectificativa': 'S',
                'ImporteRectificacion': {
                    'BaseRectificada': base_rectificada,
                    'CuotaRectificada': cuota_rectificada,
                }
            })
        return res

    def test_get_invoice_data(self):
        self._open_invoice()

        vat = self.partner.vat
        self.partner.vat = False
        with self.assertRaises(exceptions.Warning):
            self.invoice._get_sii_invoice_dict()
        self.partner.vat = vat

        invoices = self.invoice._get_sii_invoice_dict()
        test_out_inv = self._get_invoices_test('F1', '01')
        for key in invoices.keys():
            self.assertDictEqual(
                _deep_sort(invoices.get(key)),
                _deep_sort(test_out_inv.get(key)))

        self.invoice.type = 'out_refund'
        self.invoice.sii_refund_type = 'S'
        invoices = self.invoice._get_sii_invoice_dict()
        test_out_refund = self._get_invoices_test('R4', '01')
        for key in invoices.keys():
            self.assertDictEqual(
                _deep_sort(invoices.get(key)),
                _deep_sort(test_out_refund.get(key)))

        self.invoice.type = 'in_invoice'
        self.invoice.supplier_invoice_number = 'sup0001'
        invoices = self.invoice._get_sii_invoice_dict()
        test_in_invoice = self._get_invoices_test('F1', '01')
        for key in invoices.keys():
            self.assertDictEqual(
                _deep_sort(invoices.get(key)),
                _deep_sort(test_in_invoice.get(key)))

        self.invoice.type = 'in_refund'
        self.invoice.sii_refund_type = 'S'
        self.invoice.supplier_invoice_number = 'sup0001'
        invoices = self.invoice._get_sii_invoice_dict()
        test_in_refund = self._get_invoices_test('R4', '01')
        for key in invoices.keys():
            self.assertDictEqual(
                _deep_sort(invoices.get(key)),
                _deep_sort(test_in_refund.get(key)))

    def test_action_cancel(self):
        self._open_invoice()
        self.invoice.invoice_jobs_ids.state = 'started'
        self.invoice.journal_id.update_posted = True
        with self.assertRaises(exceptions.Warning):
            self.invoice.action_cancel()

    def test_sii_description(self):
        company = self.invoice.company_id
        company.write({
            'sii_header_customer': 'Test customer header',
            'sii_header_supplier': 'Test supplier header',
            'sii_description': ' | Test description',
            'sii_description_method': 'fixed',
        })
        invoice_temp = self.invoice.copy()
        self.assertEqual(
            invoice_temp.sii_description,
            'Test customer header | Test description',
        )
        invoice_temp = self.invoice.copy({'type': 'in_invoice'})
        self.assertEqual(
            invoice_temp.sii_description,
            'Test supplier header | Test description',
        )
        company.sii_description_method = 'manual'
        invoice_temp = self.invoice.copy()
        self.assertEqual(invoice_temp.sii_description, 'Test customer header')
        invoice_temp.sii_description = 'Other thing'
        self.assertEqual(invoice_temp.sii_description, 'Other thing')
        company.sii_description_method = 'auto'
        invoice_temp = self.invoice.copy()
        self.assertEqual(
            invoice_temp.sii_description, 'Test customer header | Test line',
        )

    def test_permissions(self):
        """Test permissions"""
        self.invoice.sudo(self.user).signal_workflow('invoice_open')

    def test_importe_total_when_supplier_invoice_with_irpf(self):
        # 1/ tener una factura con líneas con retención e iva
        # 2/ obtener con get_dict_in los valores a enviar al sii
        # 3/ comprobar el valor de ImporteTotal es igual al estimado
        # 1
        self._open_invoice()
        amount_base = 100
        amount_tax = 10
        amount_to_communicate_sii = amount_base + amount_tax
        invoice = self.env['account.invoice'].create({
            'partner_id': self.partner.id,
            'date_invoice': fields.Date.today(),
            'period_id': self.period.id,
            'type': 'in_invoice',
            'reference': 'PH-2020-0031',
            'supplier_invoice_number': 'PH-2020-0031',
            'account_id': self.partner.property_account_payable.id,
            'invoice_line': [
                (0, 0, {
                    'product_id': self.product.id,
                    'account_id': self.account_expense.id,
                    'account_analytic_id': self.analytic_account.id,
                    'name': 'Test line with irpf and iva',
                    'price_unit': amount_base,
                    'quantity': 1,
                    'invoice_line_tax_id': [
                        (6, 0, self.tax.ids + self.tax_irpf19.ids)],
                })],
            'sii_manual_description': '/',
        })
        # 2
        invoices = invoice._get_sii_invoice_dict()
        importe_total = invoices['FacturaRecibida']['ImporteTotal']
        # 3
        self.assertEqual(amount_to_communicate_sii, importe_total)

    def test_unlink_invoice_when_sent_to_sii(self):
        self.invoice.sii_state = 'sent'
        with self.assertRaises(exceptions.Warning):
            self.invoice.unlink()
