# Copyright 2018 Onestein (<http://www.onestein.eu>)
# License AGPL-3.0 or later (http://www.gnu.org/licenses/agpl.html).

import base64
from io import BytesIO
from zipfile import ZipFile

from odoo.tools import mute_logger
from odoo.tests.common import TransactionCase


class TestXafAuditfileExport(TransactionCase):

    def test_01_default_values(self):
        ''' Check that the default values are filled on creation '''
        record = self.env['xaf.auditfile.export'].create({})

        self.assertTrue(record)
        self.assertTrue(record.name)
        self.assertFalse(record.auditfile)
        self.assertTrue(record.auditfile_name)
        self.assertTrue(record.company_id)
        self.assertTrue(record.date_start)
        self.assertTrue(record.date_end)
        self.assertFalse(record.date_generated)
        self.assertTrue(record.fiscalyear_name)

    def test_02_export_success(self):
        ''' Do a basic auditfile export '''
        record = self.env['xaf.auditfile.export'].create({})
        record.button_generate()

        self.assertTrue(record.name)
        self.assertTrue(record.auditfile)
        self.assertTrue(record.auditfile_name)
        self.assertTrue(record.company_id)
        self.assertTrue(record.date_start)
        self.assertTrue(record.date_end)
        self.assertTrue(record.date_generated)
        self.assertTrue(record.fiscalyear_name)
        self.assertTrue(record.auditfile_success)

        zf = BytesIO(base64.b64decode(record.auditfile))
        with ZipFile(zf, 'r') as archive:
            filelist = archive.filelist
            contents = archive.read(filelist[-1]).decode('utf8')
        self.assertTrue(contents.startswith('<?xml '))

    @mute_logger(
        'odoo.addons.l10n_nl_xaf_auditfile_export.models.xaf_auditfile_export'
    )
    def test_03_export_error(self):
        ''' Failure to export an auditfile '''
        record = self.env['xaf.auditfile.export'].create({})
        record.company_id.country_id = False
        record.button_generate()

        self.assertTrue(record)
        self.assertTrue(record.name)
        self.assertFalse(record.auditfile_success)
        # still contains the faulty auditfile for debugging purposes
        self.assertTrue(record.auditfile)
        self.assertTrue(record.auditfile_name)
        self.assertTrue(record.company_id)
        self.assertTrue(record.date_start)
        self.assertTrue(record.date_end)
        self.assertTrue(record.date_generated)
        self.assertTrue(record.fiscalyear_name)

    @mute_logger(
        'odoo.addons.l10n_nl_xaf_auditfile_export.models.xaf_auditfile_export'
    )
    def test_04_invalid_characters(self):
        ''' Error because of invalid characters in an auditfile '''
        record = self.env['xaf.auditfile.export'].with_context(
            dont_sanitize_xml=True
        ).create({})
        # add an invalid character
        record.company_id.name += unichr(0x0B)
        record.button_generate()

        self.assertTrue(record)
        self.assertTrue(record.name)
        self.assertFalse(record.auditfile_success)
        self.assertTrue(record.auditfile)
        self.assertTrue(record.auditfile_name)
        self.assertTrue(record.company_id)
        self.assertTrue(record.date_start)
        self.assertTrue(record.date_end)
        self.assertTrue(record.date_generated)
        self.assertTrue(record.fiscalyear_name)
